%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MATLAB code:  Example 5.1 
% File: ENSO.m (F^{(1,2)}_{T} test statistic)
%
% Programmed by  Bruce E. Hansen.
% For updates and contact information, see the webpage:
% www.ssc.wisc.edu/~bhansen
%
% This program estimates a two-regime TAR model, and tests the null of  
% a one-regime AR against the alternative of a two-regime SETAR function. 
%
% Reference:
% Hansen, B.E. (1999). Testing for linearity.
%   Journal of Economic Surveys, 13(5), 551-576.
%   DOI: 10.1111/1467-6419.00098.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function ENSO
yname='ENSO';	                % Name of series 
load ENSOdat.txt;
dat = ENSOdat;  

global p; p = 5;	        % Order of autoregression 
global n;
global dmin_;dmin_=1;		% Minimal delay order 
global dmax_;dmax_=p;		% Maximal delay order (set to p or less)
global trim_;trim_ = .1;	% Minimal percentage of data per regime 
global qnum_;qnum_=100;	        % Number of thresholds to search. 
                                % Set qnum_=0 to search over all values 
global boot_;boot_ = 1000;	% Bootstrap replications 
global k;
global t;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Define data 
n      = length(dat(:,1));
y      = dat(p+1:n);
t      = n-p;
x      = [ones(t,1),dat(p:n-1)];
xname  = ['Constant';' Y(t-01)'];
xname2 = ['Constant  ';' Y(t-01)^2'];

j = 2;
while j<=p
    x=[x,dat(p+1-j:n-j)];
    if j<10
        kname = strcat('_Y(t-0',num2str(j),')');
    else
        kname = strcat('_Y(t-',num2str(j),')');
    end;
    xname  = [xname;kname];
    xname2 = xname;
    j=j+1;
end;

k = length(x(1,:));

% Linear Regression 
mi   = inv(x'*x);
beta = mi*(x'*y);
e    = y-x*beta;
ee   = e'*e;
xe   = x.*(e*ones(1,length(x(1,:))));
sig  = ee/t;
se   = sqrt(diag(mi*(xe'*xe)*mi));

% Conditional Variance 
e2      = e.*e;
x2      = x.*x;
m2      = inv(x2'*x2);
hetbeta = m2*(x2'*e2);
h       = x2*hetbeta;
eh      = e./sqrt(h.*(h>0)+(h<0)).*(h>0);
u       = e2-h;
x2u     = x2.*(u*ones(1,length(x(1,:))));
se2     = sqrt(diag(m2*(x2u'*x2u)*m2));
em      = e2-mean(e2)';
fh      = t*((em'*em)/(u'*u)-1);

% Report Linear Estimation 
fprintf('Dependent Variable:   %s\n',yname);
disp(' ');
disp('Linear Autoregression');
disp(' ');
hname=['Variable     ','Estimate     ','St Error'];
fprintf('%s\n',hname);
disp('----------------------------------');
for i=1:length(beta)
    fprintf('%s     %f    %f\n',xname(i,:),beta(i),se(i));
end;
disp(' ');
fprintf('Observations:                  %u\n',t);
fprintf('Sum of Squared Errors:         %f\n',ee);
fprintf('Residual Variance:             %f\n',sig);
disp(' ');
disp(' ');
disp('Conditional Variance');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=1:length(hetbeta)
    fprintf('%s     %f    %f\n',xname2(i,:),hetbeta(i),se2(i));
end;
fprintf('Heteroskedasticity F Test:     %f      %f\n',fh,1-chi2cdf(fh,p));
disp('----------------------------------');
disp(' ');
disp(' ');
disp(' ');
disp(' ');

% SETAR Estimation 
q  = qfunc(x);
qq = qsort(q);
qn = length(q(1,:));
gn = length(qq(:,1));
s  = zeros(gn,qn);
mmistore=zeros(k*(k+1)/2,qn*gn);

m = 1;
while m<=qn
    qm = q(:,m);
    j  = 1;
    while j<=gn
        gg = qq(j,m);
        if isnan(gg)
            continue;
        end;
        xd  = x.*((qm<=gg)*ones(1,length(x(1,:))));
        xxd = xd'*xd;
        xde = xd'*e;
        mmi = ginv(xxd-xxd*mi*xxd);
        indmmi = 1;
        for i=1:length(mmi(:,1))
            ii = 1;
            while ii<=i
                mmistore(indmmi,(m-1)*gn+j) = mmi(i,ii);
                indmmi = indmmi+1;
                ii     = ii+1;
            end;
        end;
        clear indmmi;
        clear ii;
        s(j,m) = xde'*mmi*xde;
        j      = j+1;
    end;
    m=m+1;
end;

[f,maxinds] = max(s);
ghat        = diag(qq(maxinds,:));

clear mmm;clear maxinds;
f       = t*(f./(ee-f));
[mf,di] = max(f);
delay   = dmin_-1+di;
gghat   = ghat(di);

d1  = (q(:,di)<=gghat);
d2  = 1-d1;
x1  = x.*(d1*ones(1,length(x(1,:))));
x2  = x.*(d2*ones(1,length(x(1,:))));
xx  = [x1,x2];
mxx = inv(xx'*xx);

betatar = mxx*(xx'*y);
etar    = y-xx*betatar;
xxe     = xx.*(etar*ones(1,length(xx(1,:))));
eetar   = etar'*etar;
sigtar  = eetar/t;

n1      = sum(d1);
n2      = sum(d2);
setar   = sqrt(diag(mxx*(xxe'*xxe)*mxx));
sig1    = (etar.^2)'*d1/n1;
sig2    = (etar.^2)'*d2/n2;
e2      = etar.^2;
d12     = [d1,d2];
z       = [d12,x(:,2:k).*x(:,2:k)];
mz      = inv(z'*z);
betaz   = mz*(z'*e2);
u       = e2-z*betaz;
zu      = z.*(u*ones(1,length(z(1,:))));
se3     = sqrt(diag(mz*(zu'*zu)*mz));
em      = e2-mean(e2)';
fz      = t*((em'*em)/(u'*u)-1);
xname3=['      D1';'      D2';xname2(2:k,:)];

% Report SETAR estimates 

disp('Threshold Autoregression');
disp(' ');
fprintf('Sum of Squared Errors:         %f\n',eetar);
fprintf('Residual Variance:             %f\n',sigtar);
fprintf('Delay Order:                   %u\n',delay);
fprintf('Threshold Estimate:            %f\n',gghat);
disp(' ');disp(' ');
disp('Regime 1 (Threshold Variable less than or equal to Threshold):');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');


for i=1:k
    fprintf('%s     %f    %f\n',xname(i,:),betatar(i),setar(i));
end;
disp(' ');
fprintf('Observations:                  %u\n',n1);
fprintf('Percentage:                    %f\n',n1/t);
fprintf('Regime Variance:              %f\n',sig1);
disp(' ');disp(' ');

disp('Regime 2 (Threshold Variable greater than Threshold):');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=k+1:2*k
    fprintf('%s     %f    %f\n',xname(i-k,:),betatar(i),setar(i));
end;
disp(' ');
fprintf('Observations:                  %u\n',n2);
fprintf('Percentage:                    %f\n',n2/t);
fprintf('Regime Variance:               %f\n',sig2);
disp(' ');disp(' ');disp(' ');

disp('Conditional Variance');
disp(' ');
fprintf('%s\n',hname);
disp('----------------------------------');
for i=1:length(betaz)
    fprintf('%s     %f    %f\n',xname3(i,:),betaz(i),se3(i));
end;
fprintf('Heteroskedasticity F Test:     %f      %f\n',fz,1-chi2cdf(fz,k));
disp('----------------------------------');
disp(' ');
disp(' ');
disp(' ');
disp(' ');

% P-Value Calculation   

% Asymptotic Bootstrap 
ab1 = zeros(boot_,p);
ab2 = zeros(boot_,p);

i = 1;
while i<=boot_;
    y1 = normrnd(0,1,t,1);
    e1 = (y1-x*mi*(x'*y1));
    s1 = zeros(gn,qn);
    y2 = e.*y1;
    e2 = (y2-x*mi*(x'*y2));
    s2 = zeros(gn,qn);
    m = 1;
    while m<=qn
        qm = q(:,m);
        j = 1;
        while j<=gn
            gg = qq(j,m);
            if isnan(gg)
                continue;
            end;
            xd  = x.*((qm<=gg)*ones(1,length(x(1,:))));
            ind = 1; % xpnd % 
           for ix=1:sqrt(2*length(mmistore(:,(m-1)*gn+j)))
               for iy=1:ix
                   mmi(ix,iy) = mmistore(ind,(m-1)*gn+j);
                   mmi(iy,ix) = mmistore(ind,(m-1)*gn+j);
                   ind        = ind+1;
               end;
           end;
           clear ix;
           clear iy;
           xde=xd'*e1;
           s1(j,m) = xde'*mmi*xde;
           xde     = xd'*e2;
           s2(j,m) = xde'*mmi*xde;
           j       = j+1;
        end;
        m = m+1;
    end;
    s1       = max(s1)';
    ab1(i,:) = s1';
    s2       = max(s2)';
    ab2(i,:) = s2'./sig;
    i        = i+1;
end;

ma1   = max(ab1')';
apv1  = mean(ab1>(ones(length(ab1(:,1)),1)*f))';
ampv1 = mean(ma1>mf)';
ma2   = max(ab2')';
apv2  = mean(ab2>(ones(length(ab2(:,1)),1)*f))';
ampv2 = mean(ma2>mf)'; 

clear mmistore;
clear xxdstore;

% Model-Based Bootstrap %
y0=dat(1:p);

% Homoskedastic Error Bootstrap 
fb=zeros(boot_,p);

i = 1;
while i<=boot_
    fb(i,:) = tar(ar_sim(y0,beta,e))';
    i       = i+1;
end;

mbt1   = max(fb')';
btpv1  = mean(fb>(ones(length(fb(:,1)),1)*f))';
btmpv1 = mean(mbt1>mf)';

% Heteroskedastic Error Bootstrap 
fb = zeros(boot_,p);

i = 1;
while i<=boot_
    fb(i,:) = tar(het_sim(y0,beta,hetbeta,eh))';
    i       = i+1;
end;
mbt2   = max(fb')';
btpv2  = mean(fb>(ones(length(fb(:,1)),1)*f))';
btmpv2 = mean(mbt2>mf)';

% Output %
fprintf('Bootstrap Replications:          %u\n',boot_);    
names=['Delay   ';'Fstat   ';'PV-A-H  ';'PV-B-H  ';'PV-A-Het';'PV-B-Het'];
f=f';
stat=[(1:p)',f,apv1,btpv1,apv2,btpv2];
disp('Delay      Fstat      PV-A-H     PV-B-H     PV-A-Het     PV-B-Het');
disp('-----------------------------------------------------------------');
for i=1:length(stat(:,1))
    fprintf('%u       %f   %f   %f   %f   %f\n',stat(i,1),stat(i,2),stat(i,3),stat(i,4),stat(i,5),stat(i,6))
end;
fprintf('Max      %f   %f   %f   %f   %f\n',mf,ampv1,btmpv1,ampv2,btmpv2);

% Density Calculation %
qm  = .99;
ub  = max([quant(ma1,qm),quant(ma2,qm),quant(mbt1,qm),quant(mbt2,qm)])';
k_x = (0:(ub/999):ub)';
k1  = (k_x.^((k/2)-1)).*exp(-k_x/2)./(gamma(k/2)*(2^(k/2)));
k1  = [k1,kernel(ma1,k_x),kernel(mbt1,k_x),kernel(ma2,k_x),kernel(mbt2,k_x)];
sun_k1 = [k_x,k1];                        % CHANGE
save('sun_k1.mat','sun_k1');


%%%%%%%%%%%%%%%%%%%%
% SETAR Procedures 
%%%%%%%%%%%%%%%%%%%%
function f=tar(dat)
global p;
global n;
global t;
y = dat(p+1:n);
x = [ones(t,1),dat(p:n-1)];

j=2;
while j<=p
    x = [x,dat(p+1-j:n-j)];
    j = j+1;
end;
q  = qfunc(x);
qq = qsort(q);
mi = ginv(x'*x);
e  = y-x*mi*(x'*y);
qn = length(q(1,:));
gn = length(qq(:,1));
s  = zeros(gn,qn);

m  = 1;
while m<=qn
    qm = q(:,m);
    j = 1;
    while j<=gn
        gg = qq(j,m);
        if isnan(gg)
            continue;
        end;
        xd  = x.*((qm<=gg)*ones(1,length(x(1,:))));
        xxd = xd'*xd;
        mmi = xxd-xxd*mi*xxd;
        xde = xd'*e;
        s(j,m) = xde'*dinv(xde,mmi);
        j      = j+1;
    end;
    m=m+1;
end;

f = max(s)';
f = t*(f./((e'*e)-f));

function qq=qsort(q)
global trim_;
global qnum_;
n  = length(q(:,1));
k  = length(q(1,:));
n1 = round(trim_*n);

if qnum_==0;
    qq = zeros(n-n1-n1+1,k);
    qq = nan*qq;
    j=1;
    while j<=k
        qj = unique(q(:,j),1);
        qj = qj(n1:(length(qj(:,1))-n1+1));
        qq(1:length(qj(:,1)),j) = qj;
        j                       = j+1;
    end;
else
    qq = zeros(qnum_,k);
    qq = nan*qq;
    j=1;
    while j<=k
        qj = unique(q(:,j));
        qj = qj(n1:(length(qj(:,1))-n1+1));
        nj = length(qj(:,1));
        if nj<=qnum_
            qq(1:nj,j) = qj;
        else
            qq(:,j)    = qj(ceil((1:qnum_)./qnum_*nj));
        end;
        j=j+1;
    end;
end;

function q=qfunc(x)
global dmin_;
global dmax_;
q = x(:,(dmin_+1):(dmax_+1));

%%%%%%%%%%%%%%%%%%%%%%%%%
% Simulation Procedures  
%%%%%%%%%%%%%%%%%%%%%%%%%
function yb=ar_sim(y0,beta,e)
global k;
t      = length(e(:,1));
u      = e(ceil(unifrnd(0,1,t,1)*t))+beta(1);
yb     = y0;
eols_c = [y0;u];
aa     = beta(2:k);

for i=(length(y0(:,1))+1):(length(eols_c(:,1)))
    datbb = eols_c(i,:);
    for j=1:length(y0(:,1))
        datbb = datbb+aa(j,:).*yb(i-j,:);
    end;
    yb = [yb;datbb];
end;

function y=het_sim(y0,beta,het,e)
global p;
t = length(e(:,1));
u = e(ceil(unifrnd(0,1,t,1)*t));
y = [y0;zeros(t,1)];

j=1;
while j<=t
    x=1;
    for i=(p+j-1):-1:j
        x=[x;y(i)];
    end;
    h      = (x.*x)'*het;
    y(p+j) = beta'*x+sqrt(h.*(h>0)).*u(j);
    j=j+1;
end;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function kern=kernel(x,b)
h    = 1.7*std(x)'/(length(x(:,1))^(.2));
g    = length(b(:,1));
kern = zeros(g,1);
i    = 1;
while i<=g;
    u = abs(b(i)-x)/h;
    kern(i) = mean((1-u.^2).*(u<=1))'*(.75)/h;
    i=i+1;
end;

function qq=quant(x,q)
s  = sortrows(x,1);
qq = s(round(length(s(:,1))*q));

function mi=ginv(m)
warning off;
lastwarn(' ');
mi = inv(m);
[mw,idw] = lastwarn;
lastwarn(' ');
warning on;
if mw(1)=='M'
   mi = pinv(m);
end;

function d=dinv(y,x)
warning off;
lastwarn(' ');
d        = (y'/x')';
[mw,idw] = lastwarn;
lastwarn(' ');
warning on;
if mw(1)=='M'
   d = ginv(x'*x)*(x'*y);
end;
